//	TorusGames3DMouse.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesMatrix44.h"
#include <float.h>
#include <math.h>


//	In DragRotate mode, the user rotates a virtual sphere.
//	The sphere should fill most of the view,
//	but not quite all of it, so that dragging near the edge
//	will rotate the sphere about the z axis.
//	The view's coordinates run from -1/2 to +1/2.
#define VIRTUAL_SPHERE_RADIUS			0.5

//	In a 3D game, at the end of a scroll, how close must
//	the tiling center be to the nearest grid point in order
//	to trigger snap-to-grid?  The tolerance gets applied
//	to each coordinate individually, independently of the others.
#define SNAP_TO_GRID_TOLERANCE			0.03125

//	In a 3D game, how long should a snap-to-grid take?
#define SCROLL_SNAP_DURATION			0.125	//	in seconds

//	In a 3D game, at the end of a rotation, how close must
//	the frame cell orientation be to the nearest
//	axis-aligned orientation in order to trigger snap-to-axis?
#define SNAP_TO_AXIS_TOLERANCE			0.999

//	In a 3D game, how long should a snap-to-axis take?
#define ROTATION_SNAP_DURATION			0.125	//	in seconds

//	Minimum drag motion required to establish scroll or slider direction.
#define MIN_ACCUMULATED_DRAG_MOTION		0.01	//	as fraction of view width


static void		MakeHitTestRay(ModelData *md, double aMouseH, double aMouseV, HitTestRay3D *aRay);
static void		TestFrameCellWallHits(HitTestRay3D *aRay);
static void		TestOneFrameCellWall(HitTestRay3D *aRay, unsigned int aDirection, double aDistance);
static void		SelectScrollAxis(ModelData *md, double aDragPoint[2], double aMotion[2]);
static void		RespondToScrollMotion(ModelData *md, double aMotion[2], double anElapsedTime);
static void		InterpretRotation(double aMousePointA[2], double aMousePointB[2], double anElapsedTime, Isometry *anIsometry, Velocity *aVelocity);
static void		FindPointOnVirtualSphere(double aMousePoint[2], double aSpherePoint[3]);
static void		RayIntersectsCubeFace(HitTestRay3D *aRay, double aCubeCenter[3], double aCubeHalfWidth, unsigned int anAxis, signed int aSign, double *aSmallestT);
static void		CoastingMomentum3DTranslation(ModelData *md, double aTimeInterval);
static void		CoastingMomentum3DRotation(ModelData *md, double aTimeInterval);


#ifdef __APPLE__
#pragma mark -
#pragma mark 3D touch
#endif

void MouseDown3D(
	ModelData	*md,
	double		aMouseH,				//	horizontal coordinate (intrinsic units rightward from center, in range [-0.5, +0.5])
	double		aMouseV,				//	 vertical  coordinate (intrinsic units   upward  from center, in range [-0.5, +0.5])
	bool		aScrollFlag,			//	Dragging with the shift key requests a scroll (desktop only)
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag)		//	Dragging in response to a flick gesture (iOS only)
{
	HitTestRay3D	theRay;
	
	//	Remember the mouse location, for use in MouseMove3D().
	md->itsPreviousDragPoint[0]	= aMouseH;
	md->itsPreviousDragPoint[1]	= aMouseV;

	//	Keep track of the accumulated drag
	//	when selecting a scrolling direction
	//	or an initial direction for the 3D Maze slider.
	md->itsAccumulated3DDragMotion[0] = 0.0;
	md->itsAccumulated3DDragMotion[1] = 0.0;
	
	//	Convert theMouseLocation to a HitTest3DRay in tiling coordinates.
	MakeHitTestRay(md, aMouseH, aMouseV, &theRay);
	
	//	Gesture handling
	//
	//	Handle two-finger and flick gestures independently
	//	of ordinary touch handling to reduce the risk of conflict
	//	when iOS reports UIGestureRecognizerStateBegan and then
	//	immediately *after* that calls -touchesCancelled:withEvent: .
	//	Confession:  some variables (its3DScrollAxisIsKnown etc.)
	//	remain shared between the gesture and ordinary touch handling code.

	if (aTwoFingerGestureFlag)
	{
		md->its3DGestureDragType	= DragScroll;
		md->its3DScrollAxisIsKnown	= false;
		md->its3DScrollSpeed		= 0.0;
	}
	else
	if (aFlickGestureFlag)
	{
		switch (md->itsViewType)
		{
			case ViewBasicLarge:
				md->its3DGestureDragType	= DragRotate;
				md->its3DRotationalVelocity	= (Velocity)VELOCITY_ZERO;
				break;
				
			case ViewRepeating:
				md->its3DGestureDragType	= DragScroll;
				md->its3DScrollAxisIsKnown	= false;
				md->its3DScrollSpeed		= 0.0;
				break;
				
			default:
				GEOMETRY_GAMES_ABORT("unexpected ViewType");
				break;
		}
	}
	else
	{
		//	Normally the game decides whether the tap or click hits an object
		//	(in which case subsequent MouseMove() and MouseUp() calls will get
		//	forwarded to the game) or not (in which case the board scrolls or rotates).
		if (md->itsSimulationStatus != SimulationNone)
		{
			//	An animation is in progress, so ignore mouse-down events
			//	to avoid interfering with the existing animation.
			md->its3DDragType = DragIgnore;
		}
		else
		if (aScrollFlag)
		{
			//	Honor the user's request to scroll.
			md->its3DDragType = DragScroll;
		}
		else	//	Ask the game whether it would like to accept a content tap.
		if (! md->itsGameIsOver	//	Ignore content hits after game is over.
		 && md->itsGame3DDragBegin != NULL
		 && (*md->itsGame3DDragBegin)(md, &theRay))
		{
			md->its3DDragType = DragContent;
		}
		else
		{
			//	Treat a generic drag...
			switch (md->itsViewType)
			{
				//	...as a rotation in ViewBasicLarge.
				case ViewBasicLarge:	md->its3DDragType = DragRotate;	break;

				//	...as a scroll in ViewRepeating.
				case ViewRepeating:		md->its3DDragType = DragScroll;	break;
				
				default:				GEOMETRY_GAMES_ABORT("unexpected ViewType");	break;
			}
		}
		
		if (md->its3DDragType == DragRotate)
		{
			md->its3DRotationalVelocity	= (Velocity)VELOCITY_ZERO;
		}

		if (md->its3DDragType == DragScroll)
		{
			md->its3DScrollAxisIsKnown	= false;
			md->its3DScrollSpeed		= 0.0;
		}
	}

	md->itsCoastingStatus = CoastingNone;

	//	Ask the idle-time routine to redraw the scene.
	md->itsChangeCount++;
}

void MouseMove3D(
	ModelData	*md,
	double		aMouseDeltaH,			//	same coordinate system as in MouseMove()
	double		aMouseDeltaV,
	double		aMouseDeltaT,			//	incremental time
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag)		//	Dragging in response to a flick gesture (iOS only)
{
	double			theOldDragPoint[2],	//	in [-0.5, +0.5] coordinates
					theNewDragPoint[2],	//	in [-0.5, +0.5] coordinates
					theMotion[2];
	HitTestRay3D	theRay;
	Isometry		theIncrementalRotation;
	
	
	theMotion[0] = aMouseDeltaH;
	theMotion[1] = aMouseDeltaV;

	theOldDragPoint[0] = md->itsPreviousDragPoint[0];
	theOldDragPoint[1] = md->itsPreviousDragPoint[1];
	
	theNewDragPoint[0] = md->itsPreviousDragPoint[0] + aMouseDeltaH;
	theNewDragPoint[1] = md->itsPreviousDragPoint[1] + aMouseDeltaV;
	
	md->itsPreviousDragPoint[0] = theNewDragPoint[0];
	md->itsPreviousDragPoint[1] = theNewDragPoint[1];
	
	MakeHitTestRay(md, theNewDragPoint[0], theNewDragPoint[1], &theRay);


	//	Gesture handling
	//
	//	Handle two-finger and flick gestures independently
	//	of ordinary touch handling to reduce the risk of conflict
	//	when iOS reports UIGestureRecognizerStateBegan and then
	//	immediately *after* that calls -touchesCancelled:withEvent: .
	//	Confession:  some variables (its3DScrollAxisIsKnown etc.)
	//	remain shared between the gesture and ordinary touch handling code.

	if (aTwoFingerGestureFlag
	 || aFlickGestureFlag)
	{
		switch (md->its3DGestureDragType)
		{
			case DragScroll:

				if ( ! md->its3DScrollAxisIsKnown )
				{
					md->itsAccumulated3DDragMotion[0] += aMouseDeltaH;
					md->itsAccumulated3DDragMotion[1] += aMouseDeltaV;

					if (fabs(md->itsAccumulated3DDragMotion[0]) >= MIN_ACCUMULATED_DRAG_MOTION
					 || fabs(md->itsAccumulated3DDragMotion[1]) >= MIN_ACCUMULATED_DRAG_MOTION)
					{
						theMotion[0] = md->itsAccumulated3DDragMotion[0];
						theMotion[1] = md->itsAccumulated3DDragMotion[1];
						md->itsAccumulated3DDragMotion[0] = 0.0;
						md->itsAccumulated3DDragMotion[1] = 0.0;

						SelectScrollAxis(md, theNewDragPoint, theMotion);	//	succeeds iff theMotion ≠ (0,0)
					}
					else
					{
						return;	//	not enough motion has accumulated
					}
				}
		
				RespondToScrollMotion(md, theMotion, aMouseDeltaT);

				break;
			
			case DragRotate:

				InterpretRotation(	theOldDragPoint,
									theNewDragPoint,
									aMouseDeltaT,
									&theIncrementalRotation,
									&md->its3DRotationalVelocity);

				ComposeIsometries(	GeometrySpherical,
									&md->its3DFrameCellIntoWorld,
									&theIncrementalRotation,
									&md->its3DFrameCellIntoWorld);

				break;
			
			default:
				GEOMETRY_GAMES_ABORT("unexpected DragType");
				break;
		}
	}
	else
	{
		//	In the 3D Maze, when the blue ball sat at a multi-way intersection,
		//	it could sometimes be very difficult to get it to move
		//	along the tube you wanted it to move along.
		//	The problem was that when you moved the mouse (or your finger) very slowly,
		//	the initial (deltaX, deltaY) would be for at most 1 pixel in some direction,
		//	for example (+1,0) or (0,-1).  If such displacements aligned more closely
		//	to undesired tubes than to the one you were trying to follow,
		//	you couldn't select the desired tube.  Of course with fast motions
		//	the 3D Maze would get displacements like (+3,-5) and it was easy
		//	to get the desired tube.  But for slow motions it was impossible.
		//	To avoid this problem, let's accumulate at least 0.01 units
		//	of mouse motion before proceeding.
		//
		//	In most situations, accumulating mouse motion makes
		//	for a jerky animation, so do it only
		//
		//		- when deciding which axis to scroll along
		//	or
		//		- when deciding which 3D Maze tube to follow
		//	or
		//		- to clear previously accumulated motion.
		//
		if ((	md->its3DDragType == DragScroll
			 && ! md->its3DScrollAxisIsKnown )
		 ||
			(	md->its3DDragType == DragContent
			 && md->itsGame == Game3DMaze
			 && md->itsGameOf.Maze3D.itsSlider.itsStatus == SliderAtNode)
		 || md->itsAccumulated3DDragMotion[0] != 0.0
		 || md->itsAccumulated3DDragMotion[1] != 0.0)
		{
			md->itsAccumulated3DDragMotion[0] += aMouseDeltaH;
			md->itsAccumulated3DDragMotion[1] += aMouseDeltaV;

			if (fabs(md->itsAccumulated3DDragMotion[0]) >= MIN_ACCUMULATED_DRAG_MOTION
			 || fabs(md->itsAccumulated3DDragMotion[1]) >= MIN_ACCUMULATED_DRAG_MOTION)
			{
				theMotion[0] = md->itsAccumulated3DDragMotion[0];
				theMotion[1] = md->itsAccumulated3DDragMotion[1];
				md->itsAccumulated3DDragMotion[0] = 0.0;
				md->itsAccumulated3DDragMotion[1] = 0.0;
			}
			else
			{
				return;	//	not enough motion has accumulated
			}
		}

		switch (md->its3DDragType)
		{
			case DragNone:
				//	Could occur if a simulation ends
				//	while an (ignored) drag is in progress.
				break;

			case DragRotate:
				InterpretRotation(	theOldDragPoint,
									theNewDragPoint,
									aMouseDeltaT,
									&theIncrementalRotation,
									&md->its3DRotationalVelocity);
				ComposeIsometries(	GeometrySpherical,
									&md->its3DFrameCellIntoWorld,
									&theIncrementalRotation,
									&md->its3DFrameCellIntoWorld);
#ifdef PREPARE_FOR_SCREENSHOT
	printf("its3DFrameCellIntoWorld = (Isometry) {%lf, %lf, %lf, %lf}\n",
		md->its3DFrameCellIntoWorld.a,
		md->its3DFrameCellIntoWorld.b,
		md->its3DFrameCellIntoWorld.c,
		md->its3DFrameCellIntoWorld.d);
#endif
				break;
			
			case DragScroll:
				if ( ! md->its3DScrollAxisIsKnown )
					SelectScrollAxis(md, theNewDragPoint, theMotion);	//	succeeds iff theMotion ≠ (0,0)
				RespondToScrollMotion(md, theMotion, aMouseDeltaT);
				break;
			
			case DragContent:
				if (md->itsGame3DDragObject != NULL)
					(*md->itsGame3DDragObject)(md, &theRay, theMotion);
				break;
			
			case DragIgnore:
				break;
		}
	}
	
	//	Ask the idle-time routine to redraw the scene.
	md->itsChangeCount++;
}

void MouseUp3D(
	ModelData	*md,
	double		aDragDuration,			//	in seconds
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag,		//	Dragging in response to a flick gesture (iOS only)
	bool		aTouchSequenceWasCancelled)
{
	double	theAngularSpeed;

	UNUSED_PARAMETER(aDragDuration);

	//	Gesture handling
	//
	//	Handle two-finger and flick gestures independently
	//	of ordinary touch handling to reduce the risk of conflict
	//	when iOS reports UIGestureRecognizerStateBegan and then
	//	immediately *after* that calls -touchesCancelled:withEvent: .
	//	Confession:  some variables (its3DScrollAxisIsKnown etc.)
	//	remain shared between the gesture and ordinary touch handling code.

	if (aTwoFingerGestureFlag
	 || aFlickGestureFlag)
	{
		switch (md->its3DGestureDragType)
		{
			case DragScroll:

				if (md->its3DScrollSpeed >   MAX_TRANSLATIONAL_COASTING_SPEED)
					md->its3DScrollSpeed =   MAX_TRANSLATIONAL_COASTING_SPEED;
				if (md->its3DScrollSpeed < - MAX_TRANSLATIONAL_COASTING_SPEED)
					md->its3DScrollSpeed = - MAX_TRANSLATIONAL_COASTING_SPEED;

				md->itsCoastingStatus = Coasting3DTranslation;

				break;
			
			case DragRotate:

				//	theAngularSpeed tells the rate of change of the half-angle θ/2,
				//	not the full rotational angle θ.  That's not a big deal,
				//	but it throws a factor of two into the value
				//	of the MAX_ROTATIONAL_COASTING_SPEED constant.

				theAngularSpeed = sqrt(md->its3DRotationalVelocity.dbdt * md->its3DRotationalVelocity.dbdt
									 + md->its3DRotationalVelocity.dcdt * md->its3DRotationalVelocity.dcdt
									 + md->its3DRotationalVelocity.dddt * md->its3DRotationalVelocity.dddt);

				if (theAngularSpeed > MAX_ROTATIONAL_COASTING_SPEED)
				{
					md->its3DRotationalVelocity.dbdt *= MAX_ROTATIONAL_COASTING_SPEED / theAngularSpeed;
					md->its3DRotationalVelocity.dcdt *= MAX_ROTATIONAL_COASTING_SPEED / theAngularSpeed;
					md->its3DRotationalVelocity.dddt *= MAX_ROTATIONAL_COASTING_SPEED / theAngularSpeed;
				}

				md->itsCoastingStatus = Coasting3DRotation;

				break;
			
			default:
				GEOMETRY_GAMES_ABORT("unexpected DragType");
				break;
		}
	}
	else
	{
		switch (md->its3DDragType)
		{
			case DragNone:
				//	Could occur if a simulation ends
				//	while an (ignored) drag is in progress.
				//	Also possible for the reason described
				//	in "Trackpad behavior" in TorusGamesGraphicsViewMac.m.
				break;

			case DragRotate:
			
				//	Here it's tempting to write code like
				//
				//		if (aTouchSequenceWasCancelled)
				//		{
				//			md->itsCoastingStatus = CoastingNone;
				//		}
				//		else
				//		{
				//			...
				//		}
				//
				//	but alas iOS 10 first handles a flick or 2-finger gesture,
				//	and then immediately afterwards cancels the plain touch sequence
				//	(the wrong order in my opinion).  So if we set
				//	md->itsCoastingStatus = CoastingNone in response to a cancelled touch sequence,
				//	we'd be canceling the coasting that the flick or 2-finger gesture
				//	had just initiated.  Instead let's let the following lines of code
				//	redundantly set itsCoastingStatus and redundantly limit the speed.
				//
				//		Caution:  This code is fragile.  It's called after
				//		the gesture has been handled.  If you ever add or change
				//		anything here, watch out for unintented side effects.
				//

				//	theAngularSpeed tells the rate of change of the half-angle θ/2,
				//	not the full rotational angle θ.  That's not a big deal,
				//	but it throws a factor of two into the value
				//	of the MAX_ROTATIONAL_COASTING_SPEED constant.

				theAngularSpeed = sqrt(md->its3DRotationalVelocity.dbdt * md->its3DRotationalVelocity.dbdt
									 + md->its3DRotationalVelocity.dcdt * md->its3DRotationalVelocity.dcdt
									 + md->its3DRotationalVelocity.dddt * md->its3DRotationalVelocity.dddt);

				if (theAngularSpeed > MAX_ROTATIONAL_COASTING_SPEED)
				{
					md->its3DRotationalVelocity.dbdt *= MAX_ROTATIONAL_COASTING_SPEED / theAngularSpeed;
					md->its3DRotationalVelocity.dcdt *= MAX_ROTATIONAL_COASTING_SPEED / theAngularSpeed;
					md->its3DRotationalVelocity.dddt *= MAX_ROTATIONAL_COASTING_SPEED / theAngularSpeed;
				}

				md->itsCoastingStatus = Coasting3DRotation;

				break;
			
			case DragScroll:
			
				//	Caution:  This code is fragile.  See comment in the DragRotate case immediately above.

				if (md->its3DScrollSpeed >   MAX_TRANSLATIONAL_COASTING_SPEED)
					md->its3DScrollSpeed =   MAX_TRANSLATIONAL_COASTING_SPEED;
				if (md->its3DScrollSpeed < - MAX_TRANSLATIONAL_COASTING_SPEED)
					md->its3DScrollSpeed = - MAX_TRANSLATIONAL_COASTING_SPEED;

				md->itsCoastingStatus = Coasting3DTranslation;

				break;
			
			case DragContent:

				if (md->itsGame3DDragEnd != NULL)
					(*md->itsGame3DDragEnd)(md, aTouchSequenceWasCancelled);

				break;
			
			case DragIgnore:
				break;
		}

		md->its3DDragType = DragNone;
	}

	//	Confession:  Because the gesture handling code and the ordinary touch handling code
	//	both make use of itsPreviousDragPoint and itsAccumulated3DDragMotion, we must
	//	avoid zeroing them out when an ordinary touch gets cancelled.  Recall that
	//	an ordinary touch gets cancelled just *after* the gesture begins, not before.
	//	I'm a little embarrassed that I'm sharing instance variables between the two
	//	code paths, but for now that's the way it is.
	if ( ! aTouchSequenceWasCancelled )
	{
		md->itsPreviousDragPoint[0]	= 0.0;
		md->itsPreviousDragPoint[1]	= 0.0;

		md->itsAccumulated3DDragMotion[0] = 0.0;	//	unnecessary, but safe (and tidy!)
		md->itsAccumulated3DDragMotion[1] = 0.0;
	}
	
	//	Ask the idle-time routine to redraw the scene.
	md->itsChangeCount++;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark 3D hit testing
#endif

static void MakeHitTestRay(
	ModelData		*md,		//	input
	double			aMouseH,	//	input, in [-1/2, +1/2] view coordinates
	double			aMouseV,
	HitTestRay3D	*aRay)		//	output
{
	double		theFrameCellIntoWorld[4][4],
				theWorldIntoFrameCell[4][4],
				theFrameCellIntoTiling[4][4];
	
	//	Initialize theRay in world coordinates.
	
	aRay->p0[0] =  0.0;
	aRay->p0[1] =  0.0;
	aRay->p0[2] = -1.0;
	aRay->p0[3] =  1.0;
	
	aRay->p1[0] = aMouseH;
	aRay->p1[1] = aMouseV;
	aRay->p1[2] = -0.5;
	aRay->p1[3] =  1.0;
	
	//	Convert theRay to frame cell coordinates.
	RealizeIsometryAs4x4MatrixInSO3(&md->its3DFrameCellIntoWorld, theFrameCellIntoWorld);
	Matrix44GeometricInverse(theFrameCellIntoWorld, theWorldIntoFrameCell);
	Matrix44RowVectorTimesMatrix(aRay->p0, theWorldIntoFrameCell, aRay->p0);
	Matrix44RowVectorTimesMatrix(aRay->p1, theWorldIntoFrameCell, aRay->p1);

	//	Set aRay's limits according to the view mode.
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			//	Find the value tMin where theRay enters the frame cell,
			//	and  the value tMax where theRay leaves the frame cell.
			aRay->tMin = DBL_MAX;
			aRay->tMax = 0.0;
			TestFrameCellWallHits(aRay);
			break;
		
		case ViewRepeating:
			aRay->tMin = 1.0;		//	near clip plane at z = -1/2
			aRay->tMax = DBL_MAX;	//	upper limit depends on how deep the game tests
			break;
		
		default:
			GeometryGamesFatalError(u"MakeHitTestRay() received unexpected ViewType.", u"Internal Error");
			break;
	}
	
	
	//	Convert theRay to tiling coordinates.
	Matrix44GeometricInverse(md->its3DTilingIntoFrameCell, theFrameCellIntoTiling);
	Matrix44RowVectorTimesMatrix(aRay->p0, theFrameCellIntoTiling, aRay->p0);
	Matrix44RowVectorTimesMatrix(aRay->p1, theFrameCellIntoTiling, aRay->p1);
}

static void TestFrameCellWallHits(
	HitTestRay3D	*aRay)
{
	TestOneFrameCellWall(aRay, 0, -0.5);
	TestOneFrameCellWall(aRay, 0, +0.5);
	TestOneFrameCellWall(aRay, 1, -0.5);
	TestOneFrameCellWall(aRay, 1, +0.5);
	TestOneFrameCellWall(aRay, 2, -0.5);
	TestOneFrameCellWall(aRay, 2, +0.5);
}

static void TestOneFrameCellWall(
	HitTestRay3D	*aRay,
	//	Example:  To test the wall at y == 0.5,
	//	pass aDirection = 1 (meaning y direction)
	//	and aDistance = 0.5.
	unsigned int	aDirection,	//	0, 1 or 2 for x, y or z direction
	double			aDistance)	//	distance in given direction
{
	unsigned int	a,
					b,
					c;
	double			s,
					t,
					theCoordB,
					theCoordC;
	
	a = aDirection;
	b = (a < 2 ? a + 1 : 0);
	c = (b < 2 ? b + 1 : 0);
	
	if (aRay->p0[a] != aRay->p1[a])
	{
		//	Use coordinate 'a' to compute
		//
		//	    aDistance - P₀[a]
		//	t = -----------------
		//	      P₁[a]   - P₀[a]
		//
		//	which tells where aRay hits the given wall.
		//
		t = (aDistance  - aRay->p0[a]) / (aRay->p1[a] - aRay->p0[a]);
		s = 1.0 - t;
		
		//	Ignore hit points that sit at or behind the observer.
		if (t <= 0.0)
			return;
		
		//	Use s and t to compute the the hit point's other two coordinates,
		//	and take their absolute value.
		theCoordB = fabs(s*aRay->p0[b] + t*aRay->p1[b]);
		theCoordC = fabs(s*aRay->p0[c] + t*aRay->p1[c]);
		
		//	Did the ray hit the wall?
		//
		//	Note:  An older draft of this program tested for hits in tiling mode
		//	taking into account the exact size of the aperture:
		//
		//		double	anAperture)	//	0.0 = fully closed, 1.0 = fully open
		//
		//		if
		//		(
		//			anAperture < 1.0					//	handle boundary cases correctly
		//		 &&
		//			(	theCoordB <= 0.5				//	hit falls within face
		//			 && theCoordC <= 0.5)
		//		 &&
		//			(	theCoordB >= 0.5 * anAperture	//	hit avoids aperture
		//			 || theCoordC >= 0.5 * anAperture)	//	(code handles anAperture == 0 correctly)
		//		)
		//
		//	The current version assumes
		//	aperture zero in fundamental domain mode (handled here) and
		//	aperture one in tiling mode (handled elsewhere).
		if (theCoordB <= 0.5	//	hit falls within face
		 && theCoordC <= 0.5)
		{
			//	The ray hits the wall.
			if (aRay->tMin > t)
				aRay->tMin = t;
			if (aRay->tMax < t)
				aRay->tMax = t;
		}
	}
	else
	{
		//	aRay does not intersect the given plane at all.
	}
}


static void SelectScrollAxis(
	ModelData	*md,
	double		aDragPoint[2],	//	in [-0.5, +0.5] view coordinates
	double		aMotion[2])		//	difference in view coordinates
{
	double			thePointInWorld[4],
					theDerivative[4][4],
					theFrameCellIntoWorld[4][4],
					theRawProjections[3][2],
					theBiggestComponent,
					theLengths[3],
					theNormalizedProjections[3][2],
					theComponents[3];
	unsigned int	i,
					j;

	//	A one-finger touch motion (or any mouse motion) will always
	//	give aMotion ≠ (0,0).  But a two-finger gesture may change
	//	without moving its center, in which case aMotion = (0,0)
	//	and we can't yet select a scroll axis.
	if (aMotion[0] == 0.0 && aMotion[1] == 0.0)
		return;

	//	Interpret aDragPoint (x,y) as a point (x, y, -1/2, 1)
	//	on the plane z == -1/2 in world space.
	thePointInWorld[0] = aDragPoint[0];
	thePointInWorld[1] = aDragPoint[1];
	thePointInWorld[2] = -0.5;
	thePointInWorld[3] = 1.0;

	//	Still working in world space, an observer at (0,0,-1)
	//	projects the scene onto the plane z = -1/2 via the function
	//
	//		              1/2      1/2
	//		p(x,y,z) = ( ----- x, ----- y, -1/2 )
	//		             z + 1    z + 1
	//
	//	Compute the partial derivatives telling how the projected point
	//	responds to movements of the original point (x,y,z).
	Compute3DProjectionDerivative(thePointInWorld, theDerivative);

	//	Convert its3DFrameCellIntoWorld from a spin vector in Spin(3)
	//	to a 4×4 matrix in SO(3), padded out with zero translational part.
	RealizeIsometryAs4x4MatrixInSO3(&md->its3DFrameCellIntoWorld, theFrameCellIntoWorld);

	//	The first three rows of theFrameCellIntoWorld express
	//	the frame cell's standard basis vectors in world coordinates.
	//	To project each one onto the plane z = -1/2, apply theDerivative.
	//
	for (i = 0; i < 3; i++)			//	which basis vector
	{
		for (j = 0; j < 2; j++)		//	which component
		{
			theRawProjections[i][j] = theFrameCellIntoWorld[i][0] * theDerivative[0][j]
									+ theFrameCellIntoWorld[i][1] * theDerivative[1][j]
									+ theFrameCellIntoWorld[i][2] * theDerivative[2][j];
		}
	}
	
	//	Initialize theBiggestComponent to zero, for use below.
	theBiggestComponent = 0.0;
	
	//	For each projected basis vector,
	for (i = 0; i < 3; i++)
	{
		//	compute its length,
		theLengths[i] = sqrt( theRawProjections[i][0] * theRawProjections[i][0]
			   				+ theRawProjections[i][1] * theRawProjections[i][1] );

		//	normalize it to length 1,
		if (theLengths[i] > 0.0)
		{
			theNormalizedProjections[i][0] = theRawProjections[i][0] / theLengths[i];
			theNormalizedProjections[i][1] = theRawProjections[i][1] / theLengths[i];
		}
		else
		{
			theNormalizedProjections[i][0] = 0.0;
			theNormalizedProjections[i][1] = 0.0;
		}
		
		//	find the component of aMotion
		//	in the projected basis vector's direction,
		theComponents[i] = fabs( aMotion[0] * theNormalizedProjections[i][0]
							   + aMotion[1] * theNormalizedProjections[i][1] );
		
		//	and check whether this is the biggest such component so far.
		if (theBiggestComponent < theComponents[i])
		{
			theBiggestComponent = theComponents[i];
			md->its3DScrollAxis	= i;
		}
	}
	
	//	The frame cell's three basis vectors are linearly independent,
	//	to their projections must span the plane, and we have found
	//	a positive biggest component.
	GEOMETRY_GAMES_ASSERT(theBiggestComponent > 0.0, "Failed to find theBiggestComponent.");
	
	//	Record the normalized projected axis for use in MouseMove().
	md->its3DProjectedScrollAxis[0] = theNormalizedProjections[md->its3DScrollAxis][0];
	md->its3DProjectedScrollAxis[1] = theNormalizedProjections[md->its3DScrollAxis][1];

	//	As MouseMove() events arrive, scale the motion along its3DScrollAxis
	//	in inverse proportion to aMotion along the projection screen.
	md->its3DScrollFactor = 1.0 / theLengths[md->its3DScrollAxis];
	
	//	Now that we've established a direction,
	//	keep it for the duration of the drag.
	md->its3DScrollAxisIsKnown = true;
}

static void RespondToScrollMotion(
	ModelData	*md,
	double		aMotion[2],
	double		anElapsedTime)
{
	double	theComponent,
			theScrollDistance;
	
	if (md->its3DScrollAxisIsKnown)
	{
		//	Technical note:  One could pre-multiply its3DScrollFactor
		//	into its3DProjectedScrollAxis, thus eliminating the need
		//	for its3DScrollFactor as a separate variable.
		//	But the present organization seems conceptually clearer,
		//	and the inefficiency is truly negligible.
		
		//	Find the component of aMotion in the direction
		//	of its3DProjectedScrollAxis.  (The latter is a unit vector.)
		theComponent = aMotion[0] * md->its3DProjectedScrollAxis[0]
					 + aMotion[1] * md->its3DProjectedScrollAxis[1];
		
		//	Scale the distance along the project axis up to a typically
		//	longer distance along the original unprojected axis.
		theScrollDistance = theComponent * md->its3DScrollFactor;
		
		//	Update the tiling's placement in the frame cell.
		md->its3DTilingIntoFrameCell[3][md->its3DScrollAxis] += theScrollDistance;

		//	Keep the tiling's center within the frame cell proper,
		//	wrapping as necessary (respecting the topology of course).
		Normalize3DTilingIntoFrameCell(md->its3DTilingIntoFrameCell, md->itsTopology);
		
		//	Compute the speed, for later use in Coasting3DTranslation.
		if (anElapsedTime > 0.0)
			md->its3DScrollSpeed = theScrollDistance / anElapsedTime;
	}
}

void Normalize3DTilingIntoFrameCell(
	double			aTilingIntoFrameCell[4][4],
	TopologyType	aTopology)
{
	double	theSwap;

	//	OK to use "if" rather than "while".
	//	All changes should be fairly small.

	if (aTilingIntoFrameCell[3][0] < -0.5)
		aTilingIntoFrameCell[3][0] += 1.0;
	
	if (aTilingIntoFrameCell[3][0] > +0.5)
		aTilingIntoFrameCell[3][0] -= 1.0;

	if (aTilingIntoFrameCell[3][1] < -0.5
	 || aTilingIntoFrameCell[3][1] > +0.5)
	{
		if (aTilingIntoFrameCell[3][1] < -0.5)
			aTilingIntoFrameCell[3][1] += 1.0;
		else
			aTilingIntoFrameCell[3][1] -= 1.0;
		
		switch (aTopology)
		{
			case Topology3DKlein:
				//	Premultiply aTilingIntoFrameCell by
				//
				//		-1  0  0  0
				//		 0  1  0  0
				//		 0  0  1  0
				//		 0  0  0  1
				//
				aTilingIntoFrameCell[0][0] = - aTilingIntoFrameCell[0][0];
				break;
			
			case Topology3DTorus:
			case Topology3DQuarterTurn:
			case Topology3DHalfTurn:
				break;
			
			default:
				GeometryGamesFatalError(u"Normalize3DTilingIntoFrameCell() received an unexpected TopologyType.", u"Internal Error");
				break;
		}
	}

	if (aTilingIntoFrameCell[3][2] < -0.5
	 || aTilingIntoFrameCell[3][2] > +0.5)
	{
		//	Account for topology while we still know whether
		//	we're increasing or decreasing aTilingIntoFrameCell[3][2].
		switch (aTopology)
		{
			case Topology3DQuarterTurn:

				//	Cells get rotated a quarter turn counterclockwise
				//	(+x axis rotating to +y axis) for each unit of depth
				//	in the +z direction.
				//	When moving forwards (z += 1), premultiply aTilingIntoFrameCell by
				//
				//		 0  1  0  0
				//		-1  0  0  0
				//		 0  0  1  0
				//		 0  0  0  1
				//
				//	When moving backwards (z -= 1),
				//	premultiply aTilingIntoFrameCell by
				//
				//		 0 -1  0  0
				//		 1  0  0  0
				//		 0  0  1  0
				//		 0  0  0  1
				//

				theSwap						= aTilingIntoFrameCell[0][0];
				aTilingIntoFrameCell[0][0]	= aTilingIntoFrameCell[1][0];
				aTilingIntoFrameCell[1][0]	= theSwap;

				theSwap						= aTilingIntoFrameCell[0][1];
				aTilingIntoFrameCell[0][1]	= aTilingIntoFrameCell[1][1];
				aTilingIntoFrameCell[1][1]	= theSwap;

				if (aTilingIntoFrameCell[3][2] < -0.5)	//	z is increasing
				{
					aTilingIntoFrameCell[1][0] = - aTilingIntoFrameCell[1][0];
					aTilingIntoFrameCell[1][1] = - aTilingIntoFrameCell[1][1];
				}
				else									//	z is decreasing
				{
					aTilingIntoFrameCell[0][0] = - aTilingIntoFrameCell[0][0];
					aTilingIntoFrameCell[0][1] = - aTilingIntoFrameCell[0][1];
				}

				break;
			
			case Topology3DHalfTurn:
				//	Premultiply aTilingIntoFrameCell by
				//
				//		-1  0  0  0
				//		 0 -1  0  0
				//		 0  0  1  0
				//		 0  0  0  1
				//
				aTilingIntoFrameCell[0][0] = - aTilingIntoFrameCell[0][0];
				aTilingIntoFrameCell[1][1] = - aTilingIntoFrameCell[1][1];
				break;
			
			case Topology3DTorus:
			case Topology3DKlein:
				break;
			
			default:
				GeometryGamesFatalError(u"Normalize3DTilingIntoFrameCell() received an unexpected TopologyType.", u"Internal Error");
				break;
		}

		if (aTilingIntoFrameCell[3][2] < -0.5)
			aTilingIntoFrameCell[3][2] += 1.0;
		else
			aTilingIntoFrameCell[3][2] -= 1.0;
	}
}

static void InterpretRotation(
	double		aMousePointA[2],	//	input,  in [-1/2, +1/2] coordinates
	double		aMousePointB[2],	//	input,  in [-1/2, +1/2] coordinates
	double		anElapsedTime,		//	input,  in seconds
	Isometry	*anIsometry,		//	output, in Spin(3), typically near (1,0,0,0)
	Velocity	*aVelocity)			//	output
{
	double	theSpherePointA[3],	//	conceptually on virtual sphere,
			theSpherePointB[3],	//		but numerically on unit sphere
			theRotationAxis[3],
			theLength,
			theAngle;
	
	FindPointOnVirtualSphere(aMousePointA, theSpherePointA);
	FindPointOnVirtualSphere(aMousePointB, theSpherePointB);
	
	//	theRotationAxis = theSpherePointA × theSpherePointB
	theRotationAxis[0] = theSpherePointA[1]*theSpherePointB[2] - theSpherePointA[2]*theSpherePointB[1];
	theRotationAxis[1] = theSpherePointA[2]*theSpherePointB[0] - theSpherePointA[0]*theSpherePointB[2];
	theRotationAxis[2] = theSpherePointA[0]*theSpherePointB[1] - theSpherePointA[1]*theSpherePointB[0];

	theLength = sqrt(theRotationAxis[0] * theRotationAxis[0]
				   + theRotationAxis[1] * theRotationAxis[1]
				   + theRotationAxis[2] * theRotationAxis[2]);
	
	if (theLength == 0.0)
	{
		*anIsometry = (Isometry)IDENTITY_ISOMETRY;
		return;
	}
	
	theRotationAxis[0] /= theLength;
	theRotationAxis[1] /= theLength;
	theRotationAxis[2] /= theLength;
	
	if (theLength <= 1.0)
		theAngle = asin(theLength);	//	assume 0 ≤ theAngle ≤ π/2
	else	//	perhaps theLength == 1.0000001
		theAngle = 0.5 * PI;
	
	//	GeometryGamesSpinGroup.c seems to use the convention
	//	that a positive rotation about an axis turns counterclockwise
	//	as seen from the axis looking towards the origin.
	//	The cross-product formula above uses the opposite convention,
	//	so negate theAngle to compensate.
	theAngle = - theAngle;
	
	anIsometry->a = cos(0.5 * theAngle);
	anIsometry->b = sin(0.5 * theAngle) * theRotationAxis[0];
	anIsometry->c = sin(0.5 * theAngle) * theRotationAxis[1];
	anIsometry->d = sin(0.5 * theAngle) * theRotationAxis[2];
	
	//	In general the derivative of
	//
	//			d(b,c,d)
	//			-------- = sin(θ/2) theRotationAxis
	//			   dt
	//	is
	//			d sin(θ/2)
	//			---------- theRotationAxis
	//			    dt
	//
	//			         1 dθ
	//		  = cos(θ/2) - -- theRotationAxis
	//			         2 dt
	//
	//	At time t = 0 that expression simplifies to
	//
	//			1 dθ
	//		  = - -- theRotationAxis
	//			2 dt
	//
	if (anElapsedTime > 0.0)
	{
		aVelocity->dbdt = 0.5 * (theAngle/anElapsedTime) * theRotationAxis[0];
		aVelocity->dcdt = 0.5 * (theAngle/anElapsedTime) * theRotationAxis[1];
		aVelocity->dddt = 0.5 * (theAngle/anElapsedTime) * theRotationAxis[2];
	}
	else
	{
		aVelocity->dbdt = 0.0;
		aVelocity->dcdt = 0.0;
		aVelocity->dddt = 0.0;
	}
}

static void FindPointOnVirtualSphere(
	double	aMousePoint[2],		//	input,  in [-1/2, +1/2] coordinates
	double	aSpherePoint[3])	//	output, conceptually on virtual sphere
								//			but numerically on unit sphere
{
	double	theLengthSquared,
			theLength;
	
	theLengthSquared = aMousePoint[0] * aMousePoint[0]
					 + aMousePoint[1] * aMousePoint[1];
	
	if (theLengthSquared <= VIRTUAL_SPHERE_RADIUS * VIRTUAL_SPHERE_RADIUS)
	{
		//	aMousePoint falls onto the virtual sphere,
		//	so map it onto the near hemisphere (z ≤ 0)...
		aSpherePoint[0] = aMousePoint[0];
		aSpherePoint[1] = aMousePoint[1];
		aSpherePoint[2] = - sqrt(VIRTUAL_SPHERE_RADIUS * VIRTUAL_SPHERE_RADIUS
								- theLengthSquared);

		//	...and then rescale the result to lie on the unit sphere. 
		aSpherePoint[0] /= VIRTUAL_SPHERE_RADIUS;
		aSpherePoint[1] /= VIRTUAL_SPHERE_RADIUS;
		aSpherePoint[2] /= VIRTUAL_SPHERE_RADIUS;
	}
	else
	{
		//	aMousePoint falls outside the virtual sphere,
		//	so map it onto the equator of the unit sphere.
		theLength		= sqrt(theLengthSquared);
		aSpherePoint[0] = aMousePoint[0] / theLength;
		aSpherePoint[1] = aMousePoint[1] / theLength;
		aSpherePoint[2] = 0.0;
	}
}


void Compute3DProjectionDerivative(
	double	aLocationInWorld[4],		//	input, last coordinate ignored
	double	aProjectionDerivative[4][4])
{
	double	x,
			y,
			z,
			zz;
	
	//	Working in world space, an observer at (0,0,-1)
	//	projects the scene onto the plane z = -1/2 via the function
	//
	//		              1/2      1/2  
	//		p(x,y,z) = ( ----- x, ----- y, -1/2 )
	//		             z + 1    z + 1
	//
	//	The partial derivatives
	//
	//		∂p       1/2
	//		-- = (  -----,        0,       0 )
	//		∂x      z + 1
	//
	//		∂p                   1/2
	//		-- = (    0,        -----,     0 )
	//		∂y                  z + 1
	//
	//		∂p      -1/2        -1/2
	//		-- = ( -------- x, -------- y, 0 )
	//		∂z     (z + 1)²    (z + 1)²
	//
	//	tell how the projected point responds to movements
	//	of the original point (x,y,z).

	x = aLocationInWorld[0];
	y = aLocationInWorld[1];
	z = aLocationInWorld[2];
	zz = 1.0 / (z + 1.0);

	aProjectionDerivative[0][0] = 0.5 * zz;
	aProjectionDerivative[0][1] = 0.0;
	aProjectionDerivative[0][2] = 0.0;
	aProjectionDerivative[0][3] = 0.0;

	aProjectionDerivative[1][0] = 0.0;
	aProjectionDerivative[1][1] = 0.5 * zz;
	aProjectionDerivative[1][2] = 0.0;
	aProjectionDerivative[1][3] = 0.0;

	aProjectionDerivative[2][0] = -0.5 * zz * zz * x;
	aProjectionDerivative[2][1] = -0.5 * zz * zz * y;
	aProjectionDerivative[2][2] = 0.0;
	aProjectionDerivative[2][3] = 0.0;

	aProjectionDerivative[3][0] = 0.0;
	aProjectionDerivative[3][1] = 0.0;
	aProjectionDerivative[3][2] = 0.0;
	aProjectionDerivative[3][3] = 0.0;
}


bool Ray3DIntersectsSphere(
	HitTestRay3D	*aRay,				//	input
	double			aSphereCenter[3],	//	input
	double			aSphereRadius,		//	input
	double			*t)					//	output, valid only when function returns true
{
	unsigned int	i;
	double			u[3],
					v[3],
					a,
					b,
					c,
					theDiscriminant;

	//	Let
	//		P(t) = P₀ + t(P₁ - P₀)
	//	
	//		Q = aSphereCenter
	//
	//		r = aSphereRadius
	//
	//	We seek a value of t for which
	//
	//		| P(t) - Q |² = r²
	//
	//	Following our nose gives
	//
	//		| P₀ + t(P₁ - P₀) - Q |² = r²
	//
	//		| (P₀ - Q) + t(P₁ - P₀) |² = r²
	//
	//	Rewriting in terms of
	//
	//		u = P₀ - Q
	//		v = P₁ - P₀
	//
	for (i = 0; i < 3; i++)
	{
		u[i] = aRay->p0[i] - aSphereCenter[i];
		v[i] = aRay->p1[i] - aRay->p0[i];
	}
	//
	//	gives
	//
	//		| u + t v |² = r²
	//
	//	which expands to
	//
	//		u·u + 2(u·v)t + (v·v)t² = r²
	//
	//	which is simply a quadratic equation
	//
	//		a t² + b t + c = 0
	//
	//	with coefficients
	//
	//		a = v₀² + v₁² + v₂²
	//		b = 2 (u₀v₀ + u₁v₁ + u₂v₂)
	//		c = u₀² + u₁² + u₂² - r²
	//
	a = v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
	b = 2.0 * (u[0]*v[0] + u[1]*v[1] + u[2]*v[2]);
	c = u[0]*u[0] + u[1]*u[1] + u[2]*u[2] - aSphereRadius*aSphereRadius;

	//	Solve a t² + b t + c = 0 using the quadratic formula
	//
	//		    -b ± sqrt(b² - 4ac)
	//		t = -------------------
	//		            2a
	//
	theDiscriminant = b*b - 4.0*a*c;
	if (theDiscriminant >= 0.0)
	{
		//	We know P₁ ≠ P₀, so
		//
		//		a = |v|² = |P₁ - P₀|² > 0
		//
		GEOMETRY_GAMES_ASSERT(a > 0, "Non-positive leading coefficient in quadratic equation");
		
		//	If both solutions fall within the allow range, report the smaller one,
		//	which corresponds to the intersection point closer to the observer.

		//	test nearer solution
		*t = (-b - sqrt(theDiscriminant))
		   / (2.0 * a);	//	a > 0 for reason shown above
		if (*t >= aRay->tMin
		 && *t <= aRay->tMax)
		{
			return true;
		}

		//	test farther solution
		*t = (-b + sqrt(theDiscriminant))
		   / (2.0 * a);	//	a > 0 for reason shown above
		if (*t >= aRay->tMin
		 && *t <= aRay->tMax)
		{
			return true;
		}

		//	Neither intersection point falls in the allowed range.
		return false;
	}
	else
	{
		//	The ray does not intersect the sphere.
		return false;
	}
}

bool Ray3DIntersectsCube(
	HitTestRay3D	*aRay,				//	input
	double			aCubeCenter[3],		//	input
	double			aCubeHalfWidth,		//	input
	double			*t)					//	output, valid only when function returns true
{
	double	theSmallestT;
	
	theSmallestT = DBL_MAX;
	
	RayIntersectsCubeFace(aRay, aCubeCenter, aCubeHalfWidth, 0, -1, &theSmallestT);
	RayIntersectsCubeFace(aRay, aCubeCenter, aCubeHalfWidth, 0, +1, &theSmallestT);
	RayIntersectsCubeFace(aRay, aCubeCenter, aCubeHalfWidth, 1, -1, &theSmallestT);
	RayIntersectsCubeFace(aRay, aCubeCenter, aCubeHalfWidth, 1, +1, &theSmallestT);
	RayIntersectsCubeFace(aRay, aCubeCenter, aCubeHalfWidth, 2, -1, &theSmallestT);
	RayIntersectsCubeFace(aRay, aCubeCenter, aCubeHalfWidth, 2, +1, &theSmallestT);
	
	if (theSmallestT < DBL_MAX)
	{
		*t = theSmallestT;
		return true;
	}
	else
	{
		return false;
	}
}

static void RayIntersectsCubeFace(
	HitTestRay3D	*aRay,				//	input
	double			aCubeCenter[3],		//	input
	double			aCubeHalfWidth,		//	input
	unsigned int	anAxis,				//	input,  0, 1 or 2 for x, y or z direction
	signed int		aSign,				//	input,  -1 or +1 for wall at x|y|z == aCubeCenter[0|1|2] ± aCubeHalfWidth
	double			*aSmallestT)		//	input and output
{
	unsigned int	a,
					b,
					c;
	double			theIntersectionPoint[3],
					s,
					t;
	
	a = anAxis;
	b = (a < 2 ? a + 1 : 0);
	c = (b < 2 ? b + 1 : 0);
	
	if (aRay->p0[a] != aRay->p1[a])
	{
		//	The cube face has equation
		//
		//		x|y|z == theIntersectionPointA
		//
		theIntersectionPoint[a] =  aCubeCenter[a]  +  aSign * aCubeHalfWidth;

		//	Use coordinate 'a' to compute
		//
		//	    IntersectionPoint[a] - P₀[a]
		//	t = ----------------------------
		//	            P₁[a]        - P₀[a]
		//
		//	which tells where aRay hits the given wall.
		//
		t = (theIntersectionPoint[a]  - aRay->p0[a]) / (aRay->p1[a] - aRay->p0[a]);
		s = 1.0 - t;
		
		//	Ignore hit points that sit outside aRay's allowed range.
		if (t < aRay->tMin || t > aRay->tMax)
			return;
		
		//	Use s and t to compute the the intersection point's other two coordinates.
		theIntersectionPoint[b] =  s * aRay->p0[b]  +  t * aRay->p1[b];
		theIntersectionPoint[c] =  s * aRay->p0[c]  +  t * aRay->p1[c];
		
		//	If
		if (
			//	the intersection points sits within the cube face and
			fabs(theIntersectionPoint[b] - aCubeCenter[b]) <= aCubeHalfWidth
		 && fabs(theIntersectionPoint[c] - aCubeCenter[c]) <= aCubeHalfWidth
			//	is closer than any previous intersection point,
		 && t < *aSmallestT)
		{
			//	then record this intersection point.
			*aSmallestT = t;
		}
	}
	else
	{
		//	aRay runs parallel to the given plane.
	}
}


#ifdef __APPLE__
#pragma mark -
#pragma mark 3D coasting
#endif

void CoastingMomentum3D(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	switch (md->itsCoastingStatus)
	{
		case Coasting3DTranslation:
			CoastingMomentum3DTranslation(md, aTimeInterval);
			break;
		
		case Coasting3DRotation:
			CoastingMomentum3DRotation(md, aTimeInterval);
			break;
		
		default:
			//	should never occur
			break;
	}
}

static void CoastingMomentum3DTranslation(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	double	theScrollDistance;

	//	its3DScrollSpeed may be positive or negative
	//	according to which way the fundamental cube is scrolling.
	//	Naturally the deceleration always acts opposite to the motion.
	if (md->its3DScrollSpeed > 0.0)
	{
		md->its3DScrollSpeed -= TRANSLATIONAL_COASTING_DECELERATION * aTimeInterval;
		if (md->its3DScrollSpeed < 0.0)
			md->its3DScrollSpeed = 0.0;
	}
	if (md->its3DScrollSpeed < 0.0)
	{
		md->its3DScrollSpeed += TRANSLATIONAL_COASTING_DECELERATION * aTimeInterval;
		if (md->its3DScrollSpeed > 0.0)
			md->its3DScrollSpeed = 0.0;
	}
	
	if (md->its3DScrollSpeed != 0.0)
	{
		theScrollDistance = md->its3DScrollSpeed * aTimeInterval;
		md->its3DTilingIntoFrameCell[3][md->its3DScrollAxis] += theScrollDistance;
		Normalize3DTilingIntoFrameCell(md->its3DTilingIntoFrameCell, md->itsTopology);
	}
	else
	{
		md->itsCoastingStatus = CoastingNone;

		//	Snap to grid, if we can do so without
		//	distrubing a pre-existing simulation.
		if (md->itsSimulationStatus == SimulationNone)
			SnapToGrid3D(md, SNAP_TO_GRID_TOLERANCE, SCROLL_SNAP_DURATION);
	}
}

static void CoastingMomentum3DRotation(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	double		theOldAngularSpeed,
				theNewAngularSpeed;
	Isometry	theIncrementalRotation;

	//	The derivative
	//
	//			d(b,c,d)   1 dθ
	//			-------- = - -- theRotationAxis
	//			   dt      2 dt
	//
	//	gives the rate of change of the half-angle θ/2,
	//	not the full rotational angle θ.  That's not a big deal,
	//	but it throws a factor of two into the value
	//	of the ROTATIONAL_COASTING_DECELERATION constant.
	//
	theOldAngularSpeed = sqrt(md->its3DRotationalVelocity.dbdt * md->its3DRotationalVelocity.dbdt
							+ md->its3DRotationalVelocity.dcdt * md->its3DRotationalVelocity.dcdt
							+ md->its3DRotationalVelocity.dddt * md->its3DRotationalVelocity.dddt);
	if (theOldAngularSpeed > 0.0)
	{
		theNewAngularSpeed = theOldAngularSpeed
						   - ROTATIONAL_COASTING_DECELERATION * aTimeInterval;
		if (theNewAngularSpeed < 0.0)
			theNewAngularSpeed = 0.0;
		
		md->its3DRotationalVelocity.dbdt *= theNewAngularSpeed / theOldAngularSpeed;
		md->its3DRotationalVelocity.dcdt *= theNewAngularSpeed / theOldAngularSpeed;
		md->its3DRotationalVelocity.dddt *= theNewAngularSpeed / theOldAngularSpeed;
	}
	else
	{
		theNewAngularSpeed = 0.0;

		md->its3DRotationalVelocity.dbdt = 0.0;
		md->its3DRotationalVelocity.dcdt = 0.0;
		md->its3DRotationalVelocity.dddt = 0.0;
	}
	
	if (theNewAngularSpeed > 0.0)
	{
		IntegrateOverTime(	GeometrySpherical,
							&md->its3DRotationalVelocity,
							aTimeInterval,
							&theIncrementalRotation);

		ComposeIsometries(	GeometrySpherical,
							&md->its3DFrameCellIntoWorld,
							&theIncrementalRotation,
							&md->its3DFrameCellIntoWorld);
	}
	else
	{
		md->itsCoastingStatus = CoastingNone;

		//	Snap to axes, if we can do so without
		//	distrubing a pre-existing simulation.
		if (md->itsSimulationStatus == SimulationNone)
			SnapToAxes3D(md, SNAP_TO_AXIS_TOLERANCE, ROTATION_SNAP_DURATION, false);
	}
}
